<?php
/**
 * NOTICE OF LICENSE
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2017 D3 Data Development
 * @license   LICENSE.txt
 */

ini_set('error_reporting', E_ALL ^ E_NOTICE);

use d3\heidelpay\blowfish\Blowfish;
use d3\heidelpay\response\Parser;
use Symfony\Component\HttpFoundation\Response;

require_once(dirname(__FILE__).'/heidelpay.php'); // Base Controller

/**
 * Class HeidelpayNotifyModuleFrontController
 */
class HeidelpayNotifyModuleFrontController extends HeidelpayModuleFrontController
{
    /**
     *
     */
    const REDIRECT = 'redirect';

    /**
     * @var
     */
    public $currentTransactionId;

    /**
     * @var string
     */
    public $currentTemplate = 'module:heidelpay/views/templates/front/notify.tpl';

    /**
     * @var
     */
    protected $redirectUrl;

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function initContent()
    {
        $this->setRedirectUrl($this->context->link->getBaseLink());

        $response = $this->getResponse();

        PrestaShopLogger::addLog(
            __METHOD__ . '::' . __LINE__ . ' called - response = ' . var_export($response, true),
            1,
            null,
            'Heidelpay Module',
            null,
            true
        );

        //file_put_contents(__DIR__ . '/notify.log', var_export($response, true), FILE_APPEND);
        $response = preg_replace('/<Criterion(\s+)name="(\w+)">(.+)<\/Criterion>/', '<$2>$3</$2>', $response);

        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($response);
        if (false === $xml) {
            $xmlErrors = '';
            foreach (libxml_get_errors() as $error) {
                $xmlErrors .= $error->message;
                $xmlErrors .= '__@@';
            }

            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__ . '::xml errors occured at notification: ' . $xmlErrors,
                3,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        $processingCode = (string)$xml->Transaction->Processing['code'];
        $transaction    = HeidelpayTransactionModel::loadFromUniqueID($xml->Transaction->Identification->UniqueID);

        if ($transaction instanceof HeidelpayTransactionModel
            && false == empty($transaction->id_heidelpayTransactions)
        ) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__ . ' Transaction already in DB: ' . $transaction->id_heidelpayTransactions,
                1,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT, Response::HTTP_OK);

            return;
        }

        $blowFish                    = new Blowfish(HeidelpayModuleFrontController::BLOFISH_KEY);
        $criterionSession            = $xml->Transaction->Analysis->SESSION;
        $aTmp                        = explode("__@@", $criterionSession);
        $idCustomer                  = $aTmp[2];
        $xmlRecord                   = new HeidelpayTransactionModel();
        $xmlRecord->id_transaction   = $xml->Transaction->Identification->UniqueID;
        $xmlRecord->id_customer      = $idCustomer;
        $xmlRecord->transaction_data = $blowFish->encrypt($response);

        try {
            $result = $xmlRecord->add();
        } catch (PrestaShopDatabaseException $prestaShopDatabaseException) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__
                . ':: could not save transaction to database: ' . $prestaShopDatabaseException->getMessage(),
                2,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        } catch (PrestaShopException $prestaShopException) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__
                . ':: could not save transaction to database: ' . $prestaShopException->getMessage(),
                2,
                null,
                'Heidelpay Module',
                null,
                true
            );

            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        if (false == $result) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__ . ':: could not save transaction to database',
                2,
                null,
                'Heidelpay Module',
                null,
                true
            );

            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        $referenceId = (string)$xml->Transaction->Identification->ReferenceID;

        $transactionID = $this->getTransactionID($referenceId);
        if (false == $transactionID) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__ . '::reference not found: ' . $referenceId,
                3,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        }
        $activeRecord                   = new HeidelpayTransactionModel($transactionID);
        $xmlRecord->id_transactiongroup = $activeRecord->id_transactiongroup;
        $xmlRecord->save();

        if (false == isset($xml->Transaction->Processing['code'])) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::' . __LINE__ . '::xml payment code not found.',
                3,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        $processingMessage = ((string)$xml->Transaction->Processing->Return);

        if ('PP.RC.90.00' != $processingCode) {
            PrestaShopLogger::addLog(
                __METHOD__ . "::xml wrong payment code found (only PP.RC.90.00):"
                . " {$processingCode}::{$processingMessage}",
                3,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        $order = new Order((int)$activeRecord->id_transactiongroup);
        if ($order->id !== (int)$activeRecord->id_transactiongroup) {
            PrestaShopLogger::addLog(
                __METHOD__ . '::order not found: ' . $activeRecord->id_transactiongroup,
                3,
                null,
                'Heidelpay Module',
                null,
                true
            );
            $this->exitOrRedirect(self::REDIRECT);

            return;
        }

        $currency = new Currency($order->id_currency);

        // TODO: check if float than cast it (float)
        $order->addOrderPayment(
            (float)$xml->Transaction->Payment->Clearing->Amount,
            $order->payment,
            $xml->Transaction->Identification->ShortID,
            $currency,
            null,
            null
        );

        if ($this->isOrderAmountReached($order)) {
            $order->setCurrentState(Configuration::get('HEIDELPAY_STATE_PREPAYMENT'));
            $order->save();
        }

        if ($this->isToMuchOrderAmountReached($order)) {
            $order->setCurrentState(Configuration::get('PS_OS_ERROR'));
            $order->save();
        }
    }

    /**
     * @return bool|string
     */
    protected function getResponse()
    {
        return Tools::file_get_contents('php://input');
    }

    /**
     * @param string $sReturn
     * @param int    $statuscode
     */
    protected function exitOrRedirect($sReturn = "", $statuscode = Response::HTTP_NO_CONTENT)
    {
        if ($sReturn === self::REDIRECT) {
            $response = new Response('', $statuscode);
            $response->send();
        } else {
            echo $this->getRedirectUrl();
        }
        die;
    }

    /**
     * returns sRedirectUrl
     *
     * @return string
     */
    public function getRedirectUrl()
    {
        return $this->redirectUrl;
    }

    /**
     * set property sRedirectUrl and add stoken parameter
     *
     * @param $sReturnURL
     *
     * @return $this
     */
    public function setRedirectUrl($sReturnURL)
    {
        $this->redirectUrl = $sReturnURL;

        return $this;
    }

    /**
     * @param $referenceId
     *
     * @return false|null|string
     */
    protected function getTransactionID($referenceId)
    {
        $sql = new DbQuery();
        $sql->select('id_heidelpayTransactions');
        $sql->from('heidelpayTransactions');
        $sql->where('id_transaction = \''.pSQL($referenceId).'\'');

        return Db::getInstance()->getValue($sql);
    }

    /**
     * @param Order $order
     *
     * @return bool
     */
    protected function isOrderAmountReached(Order $order)
    {
        $amount = 0.00;
        foreach ($order->getOrderPayments() as $orderPayment) {
            /** @var $orderPayment OrderPayment */
            $amount += (float)$orderPayment->amount;
        }
        $orderTotalPaid = $order->getOrdersTotalPaid();
        $orderTotalPaid = Tools::ps_round($orderTotalPaid, 2);

        return $orderTotalPaid === $amount;
    }

    /**
     * @param Order $order
     *
     * @return bool
     */
    protected function isToMuchOrderAmountReached(Order $order)
    {
        $amount = 0.00;
        foreach ($order->getOrderPayments() as $orderPayment) {
            /** @var $orderPayment OrderPayment */
            $amount += (float)$orderPayment->amount;
        }
        $orderTotalPaid = $order->getOrdersTotalPaid();
        $orderTotalPaid = Tools::ps_round($orderTotalPaid, 2);

        return $orderTotalPaid < $amount;
    }

    /**
     * process 3d secure data after step 4, returns next action-url
     *
     * @param Parser $parser
     *
     * @return string
     */
    protected function processThreeDSecure(Parser $parser)
    {

        // get payment type
        $query = array('currenttransactionid' => $this->currentTransactionId);

        $baseUri = $this->context->link->getModuleLink(
            'heidelpay',
            $parser->getActualTransactionData()->getInputposition(),
            $query
        );

        if ($parser->isThreeDSecureSuccess()) {
            $baseUri = $this->context->link->getModuleLink('heidelpay', 'validation', $query);
        }

        $this->setRedirectUrl($baseUri);

        return $this->getRedirectUrl();
    }

    /**
     * process input step 4 returns next action/url
     *
     * @param Parser $parser
     *
     * @return string
     */
    protected function processResponse(Parser $parser)
    {

        // get payment type
        $query = array('currenttransactionid' => $this->currentTransactionId);

        $controller = $parser->getActualTransactionData()->getInputposition();

        if ($parser->isWaiting()) {
            $query['3dsecure'] = 1;
        } elseif ($parser->isSimpleSuccess()) {
            $controller = 'validation';
        }

        $baseUri = $this->context->link->getModuleLink('heidelpay', $controller, $query);

        $this->setRedirectUrl($baseUri);

        return $this->getRedirectUrl();
    }
}
