<?php
/**
 * NOTICE OF LICENSE
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2017 D3 Data Development
 * @license   LICENSE.txt
 */

require_once('HeidelpayBasketItemValueModel.php');
require_once('HeidelpayCartRuleValueModel.php');
require_once('HeidelpayShippingValueModel.php');
require_once('HeidelpayGiftWrappingValueModel.php');

/**
 * Class HeidelpayBasketValueModel
 */
class HeidelpayBasketValueModel
{

    /**
     * @var
     */
    public $amountTotalNetto;

    /**
     * @var
     */
    public $amountTotalVat;

    /**
     * @var
     */
    public $amountTotalDiscount;

    /**
     * @var array
     */
    public $basketItems = array();

    /**
     * @var int
     */
    protected $position = 1;

    public $currencyCode;

    public $itemCount;

    /**
     * HeidelpayBasketValueModel constructor.
     *
     * @param Cart $cart
     *
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function __construct(Cart $cart)
    {
        $currency = new Currency($cart->id_currency);

        $summaryDetails            = $cart->getSummaryDetails();
        $this->amountTotalNetto    = $summaryDetails['total_price_without_tax'];
        $this->amountTotalVat      = $summaryDetails['total_tax'];
        $this->amountTotalDiscount = $summaryDetails['total_discounts'];
        $this->currencyCode        = $currency->iso_code;
        $this->itemCount           = $this->getItemCount($cart);
        $this->basketItems         = $this->getBasketItems($cart);
    }

    /**
     * @param Cart $cart
     *
     * @return int
     */
    protected function getItemCount(Cart $cart)
    {
        $itemCount      = 0;
        $summaryDetails = $cart->getSummaryDetails();
        $itemCount      += count($summaryDetails['products']);
        $itemCount      += count($summaryDetails['discounts']);
        if ($summaryDetails['total_shipping']
            && isset($summaryDetails['carrier'])
            && $summaryDetails['carrier'] instanceof Carrier
        ) {
            $itemCount++;
        }

        if ($cart->getGiftWrappingPrice() > 0) {
            $itemCount++;
        }

        return $itemCount;
    }

    /**
     * @param Cart $cart
     *
     * @return array
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    protected function getBasketItems(Cart $cart)
    {
        $basketItems = array();

        foreach ($cart->getProducts() as $product) {
            $basketItems[] = new HeidelpayBasketItemValueModel($product, $this->position);
            $this->position++;
        }

        foreach ($cart->getCartRules() as $cartRule) {
            $basketItems[] = new HeidelpayCartRuleValueModel($cartRule, $this->position);
            $this->position++;
        }

        $default_country = null;
        if (!$cart->isMultiAddressDelivery()
            && isset($cart->id_address_delivery) // Be carefull, id_address_delivery is not usefull one 1.5
            && $cart->id_address_delivery
            && Customer::customerHasAddress($cart->id_customer, $cart->id_address_delivery)
        ) {
            $address = Address::initialize((int) $cart->id_address_delivery);
            $default_country = new Country($address->id_country);
        }
        $delivery_option = $cart->getDeliveryOption($default_country, false, false);

        $delivery_option_list = $cart->getDeliveryOptionList($default_country);
        foreach ($delivery_option as $id_address => $key) {
            if (!isset($delivery_option_list[$id_address])
                || !isset($delivery_option_list[$id_address][$key])
                || empty($delivery_option_list[$id_address][$key]['total_price_with_tax'])
            ) {
                continue;
            }
            $basketItems[] = new HeidelpayShippingValueModel($delivery_option_list[$id_address][$key], $this->position);
            $this->position++;
        }

        if ($cart->getGiftWrappingPrice() > 0) {
            $giftWrappingData = array(
                'priceWithTax'    => $cart->getGiftWrappingPrice(true),
                'priceWithoutTax' => $cart->getGiftWrappingPrice(false),
            );

            $basketItems[] = new HeidelpayGiftWrappingValueModel($giftWrappingData, $this->position);
            $this->position++;
        }

        return $basketItems;
    }
}
