<?php
/**
 * NOTICE OF LICENSE
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2017 D3 Data Development
 * @license   LICENSE.txt
 */

use d3\heidelpay\Ngw;
use d3\heidelpay\ngw\Request;
use d3\heidelpay\payment\Creditcard;
use d3\heidelpay\request\Config;
use d3\heidelpay\transactionlog\reader\Transaction;
use d3\heidelpay\blowfish\Blowfish;

require_once(dirname(__FILE__).'/heidelpay.php'); // Base Controller

/**
 * Class HeidelpayCreditcardModuleFrontController
 */
class HeidelpayCreditcardModuleFrontController extends HeidelpayModuleFrontController
{
    /**
     * @var
     */
    public $isHeidelpayDebugMode;

    /**
     * @var string
     */
    public $currentTemplate = 'module:heidelpay/views/templates/front/hpf.tpl';

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     * @throws Exception
     */
    public function initContent()
    {
        $this->isHeidelpayDebugMode = (bool)Configuration::get('HEIDELPAY_LIVE_MODE');
        $this->context->smarty->assign('isHeidelpayDebugMode', $this->isHeidelpayDebugMode);
        $currentTransactionId = Tools::getValue('currenttransactionid');
        if ($currentTransactionId //
            && HeidelpayTransactionModel::existsInDatabase($currentTransactionId, "heidelpayTransactions")) {
            $blowFish = new Blowfish(HeidelpayModuleFrontController::BLOFISH_KEY);

            $activeRecord    = new HeidelpayTransactionModel($currentTransactionId);
            $transactionData = $this->decode($blowFish->decrypt($activeRecord->transaction_data));

            $transaction = new Transaction();
            $transaction->read($transactionData);
            if ($transaction->getResult() === 'NOK' || false == $this->isHashValid($transactionData)) {
                $this->setErrorTemplate($transaction->getReturncode(), false);
            }
        }

        $formularParameters = $this->getFormularParameter();
        $result             = $this->validateCustomerData($formularParameters);

        if (false == empty($result)) {
            $this->errors = $result;
            $this->setTemplate("module:heidelpay/views/templates/front/errorpage.tpl");
            parent::initContent();

            return;
        }

        $data = $this->getDataCreditCard();

        $request = new Request();

        $request->setUrl($this->getHeidelpayUrl());
        $request->setCurlTimeOut($this->getHeidelpayCurlTimeOut());
        $request->setData($data);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-request data: '.var_export($data, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        $ngw      = new Ngw();
        $response = $ngw->sendPostRequest($request);

        PrestaShopLogger::addLog(
            'HeidelpayResponse::initContent::NGW-response data: '.var_export($response, true),
            1,
            null,
            'Heidelpay Module',
            (int)$this->context->cart->id,
            true
        );

        $this->context->smarty->assign('formularParameters', $formularParameters);

        if (isset($response['FRONTEND_PAYMENT_FRAME_URL']) && 'ACK' === $response['PROCESSING_RESULT']) {
            $transaction = new Transaction();
            $transaction->read($response);
            $redirectUrl = $transaction->getRedirecturl();
            if (false == empty($redirectUrl)) {
                if (false == substr_count($redirectUrl, "?")) {
                    $redirectUrl .= "?";
                }

                foreach ($transaction->getRedirectparameters() as $index => $value) {
                    $redirectUrl .= $index."=".rawurlencode($value).'&';
                }
                $this->context->smarty->assign('iframeUrl', $redirectUrl);
                $this->registerStylesheet(
                    'modules-heidelpay_front',
                    'modules/'.$this->module->name.'/views/css/front.css',
                    array('media' => 'all', 'priority' => 150)
                );
            }
        } else {
            $this->setErrorTemplate($response['PROCESSING_RETURN_CODE']);
        }
        parent::initContent();
    }

    /**
     * @return array
     * @throws Exception
     */
    protected function getDataCreditCard()
    {
        $creditcard = new Creditcard();

        $config = new Config();
        $config->setIsTestmode(!Configuration::get('HEIDELPAY_LIVE_MODE'));
        $config->setResponseUrl($this->context->link->getModuleLink('heidelpay', 'response'));

        // TODO: RG moeglich ?
        $paymentCode = $creditcard->getPaymentCode().".".Configuration::get('HEIDELPAY_ACCOUNT_TRANSTYPE_CC');

        $data                                  = array();
        $data['REQUEST.VERSION']               = "1.0";
        $data['SECURITY.SENDER']               = Configuration::get('HEIDELPAY_ACCOUNT_SECURITYSENDER');
        $data['FRONTEND.RESPONSE_URL']         = $config->getResponseUrl();
        $data['FRONTEND.MODE']                 = "WHITELABEL";
        $data['FRONTEND.ENABLED']              = "true";
        $data['FRONTEND.LANGUAGE']             = $this->context->language->iso_code;
        $data['FRONTEND.PAYMENT_FRAME_ORIGIN'] = Tools::getShopDomainSsl(true);
        $data['IDENTIFICATION.TRANSACTIONID']  = rand()."__@@".date('Ymd-Hms');
        $data['USER.LOGIN']                    = Configuration::get('HEIDELPAY_ACCOUNT_LOGIN');
        $data['USER.PWD']                      = Configuration::get('HEIDELPAY_ACCOUNT_PASSWORD');
        $data['TRANSACTION.MODE']              = $config->getTransactionMode();
        $data['TRANSACTION.CHANNEL']           = Configuration::get('HEIDELPAY_ACCOUNT_CHANNEL_CC');
        $data['PAYMENT.CODE']                  = $paymentCode;
        $data['PRESENTATION.AMOUNT']           = $this->context->cart->getOrderTotal();
        $data['PRESENTATION.CURRENCY']         = $this->context->currency->iso_code;
        $data['SHOP.TYPE']                     = "PrestaShop "._PS_VERSION_;
        $data['SHOPMODULE.VERSION']            = $this->module->version;
        $checksum                              = $this->context->cookie->getFamily('checksum')['checksum'];
        $data['CRITERION.SESSION']             = "creditcard__@@".$checksum.'__@@'.$this->context->customer->id;
        $data['CRITERION.HASH']                = $this->generateHash($data);
        $data['CONTACT.IP']                    = Tools::getRemoteAddr();

        return $data;
    }

    /**
     * @return array
     */
    public function getFormularParameter()
    {
        $billing_address            = new Address($this->context->cart->id_address_invoice);
        $data                       = array();
        $data['ADDRESS.STREET']     = trim("{$billing_address->address1} {$billing_address->address2}");
        $data['ADDRESS.ZIP']        = $billing_address->postcode;
        $data['ADDRESS.CITY']       = $billing_address->city;
        $data['NAME.GIVEN']         = $this->context->customer->firstname;
        $data['NAME.FAMILY']        = $this->context->customer->lastname;
        $data['ADDRESS.COUNTRY']    = $this->context->country->iso_code;
        $data['NAME.COMPANY']       = $this->context->customer->company;
        $data['NAME.BIRTHDATE']     = $this->context->customer->birthday;
        $data['CONTACT.EMAIL']      = $this->context->customer->email;
        $data['CONTACT.IP']         = Tools::getRemoteAddr();
        $data['SHOP.TYPE']          = _PS_VERSION_;
        $data['SHOPMODULE.VERSION'] = $this->module->version;
        $checksum                   = $this->context->cookie->getFamily('checksum')['checksum'];
        $data['CRITERION.SESSION']  = "creditcard__@@".$checksum.'__@@'.$this->context->customer->id;
        $data['CRITERION.HASH']     = $this->generateHash($data);

        return $data;
    }
}
