<?php
/**
 * NOTICE OF LICENSE
 * This file is licenced under the Software License Agreement.
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 * You must not modify, adapt or create derivative works of this source code
 *
 * @author    D3 Data Development
 * @copyright 2017 D3 Data Development
 * @license   LICENSE.txt
 */

use d3\heidelpay\blowfish\Blowfish;

/**
 * Class HeidelpayTransactionModel
 */
class HeidelpayTransactionModel extends ObjectModel
{
    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table'   => 'heidelpayTransactions',
        'primary' => 'id_heidelpayTransactions',
        'fields'  => array(
            'id_transaction'      => array(
                'type'     => self::TYPE_STRING,
                'validate' => 'isGenericName',
                'required' => true,
                'size'     => 255,
            ),
            'id_transactiongroup' => array(
                'type'     => self::TYPE_STRING,
                'validate' => 'isGenericName',
                'size'     => 255,
            ),
            'id_customer'         => array(
                'type' => self::TYPE_INT,
                'size' => 15,
            ),
            'transaction_data'    => array(
                'type'     => self::TYPE_HTML,
                'required' => true,
            ),
        ),
    );

    /**
     * @var
     */
    public $id_transaction;

    /**
     * @var
     */
    public $transaction_data;

    /**
     * @var
     */
    public $id_heidelpayTransactions;

    /**
     * @var
     */
    public $id_customer;

    /**
     * @var
     */
    public $id_transactiongroup;

    /**
     * @var
     */
    public $timestamp;

    /**
     * @return bool
     */
    public static function createTables()
    {
        $sql = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'heidelpayTransactions` (
			`id_heidelpayTransactions` INT(15) NOT NULL AUTO_INCREMENT,
			`id_transaction` VARCHAR(255) NOT NULL,
			`id_transactiongroup` VARCHAR(255) NOT NULL,
			`id_customer` INT(15) NOT NULL,
			`transaction_data` TEXT NOT NULL,
			`timestamp` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (`id_heidelpayTransactions`),
            UNIQUE KEY (`id_transaction`)
			) ENGINE = '._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8';

        return Db::getInstance()->execute($sql);
    }

    /**
     * @return bool
     */
    public static function dropTables()
    {
        $sql = 'DROP TABLE IF EXISTS `'._DB_PREFIX_.'heidelpayTransactions`';

        return Db::getInstance()->execute($sql);
    }

    /**
     * @param $orderId
     * @param $paymentCode
     *
     * @return bool
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function loadFromOrderID($orderId, $paymentCode)
    {
        $transactions = HeidelpayTransactionModel::hydrateCollection(
            'HeidelpayTransactionModel',
            Db::getInstance()->executeS(
                'SELECT * FROM `' . _DB_PREFIX_ . 'heidelpayTransactions` ' //
                . 'WHERE `id_transactiongroup` = ' . (int)($orderId)
            )
        );

        $blowFish = new Blowfish(HeidelpayModuleFrontController::BLOFISH_KEY);
        foreach ($transactions as $transaction) {
            $transactionData = $blowFish->decrypt($transaction->transaction_data);
            $Reader          = new d3\heidelpay\transactionlog\reader\Transaction();
            $Reader->read($transactionData);
            if ($Reader->getPaymentcode() == $paymentCode) {
                return $transaction;
            }
        }

        return false;
    }

    /**
     * @param $uniqueId
     *
     * @return HeidelpayTransactionModel|false
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public static function loadFromUniqueID($uniqueId)
    {
        if (empty($uniqueId)) {
            return false;
        }
        $dbInstance = Db::getInstance();

        $transactions = HeidelpayTransactionModel::hydrateCollection(
            'HeidelpayTransactionModel',
            $dbInstance->executeS(
                'SELECT * FROM `' . _DB_PREFIX_ . 'heidelpayTransactions` ' //
                . 'WHERE `id_transaction` = \'' . ($dbInstance->escape($uniqueId)) . '\''
            )
        );

        return current($transactions);
    }
}
