<?php
namespace d3\heidelpay\request;

use d3\heidelpay\payment\Billsafe;
use d3\heidelpay\payment;
use d3\heidelpay\payment\data\Position;
use d3\heidelpay\request\exception\InvalidheidelpaydataException;
use d3\heidelpay\request\exception\InvalidTagNameException;
use d3\heidelpay\request\exception\NoTransactionUniqueId;
use d3\heidelpay\request\exception\UnknownheidelpayonlinetransferpaymentException;
use d3\heidelpay\request\pattern;
use d3\heidelpay\settings\exception\Emptypasswordexception;
use d3\heidelpay\settings\exception\emptysecuritysenderexception;
use d3\heidelpay\request\exception\NoBasketException;
use d3\heidelpay\settings\exception\Emptyusernameexception;

/**
 * Class Xml
 */
class Xml
{
    /**
     *
     */
    const XML_BODY_PLACEHOLDER = '{{requestbody}}';

    /** @var Config */
    protected $configuration;

    /** @var  payment\Data */
    protected $currentPaymentData;

    /**
     * Default constructor, sets properties
     *
     * @param Config $configuration
     */
    public function __construct(Config $configuration)
    {
        $this->configuration = $configuration;
    }

    /**
     * Returns indentification xml tag
     *
     * @return string
     */
    public function getXmlTagIdentification()
    {
        $tag = <<<XML
<Identification>
    {$this->getXmlTagTransactionID()}
    {$this->getXmlTagReferenceID()}
</Identification>
XML;

        return $tag;
    }

    /**
     * returns the TransactionID xml tag
     *
     * @return string
     */
    public function getXmlTagTransactionID()
    {
        $sTransactionId = $this->currentPaymentData->getTransactionId();
        $tag            = <<<XML
<TransactionID>{$sTransactionId}</TransactionID>
XML;

        return $tag;
    }

    /**
     * returns the ReferenceID xml tag
     *
     * @return null|string
     * @throws NoTransactionUniqueId
     */
    public function getXmlTagReferenceID()
    {
        $referenceId = $this->currentPaymentData->getReferenceId();

        if (empty($referenceId)) {
            return '';
        }
        $tag = <<<XML
<ReferenceID>{$referenceId}</ReferenceID>
XML;

        return $tag;
    }

    /**
     * returns the Payment xml tag
     *
     * @return string
     */
    public function getXmlTagPayment()
    {
        $sMemoTag = $this->getXmlTagMemo();

        $paymentOption = $this->currentPaymentData->getPaymentCombination();
        if (is_null($sMemoTag)) {
            $tag = <<<XML
<Payment code="{$paymentOption}">
    {$this->getXmlTagPresentation()}
</Payment>
XML;

            return $tag;
        }

        $tag = <<<XML
<Payment code="{$paymentOption}">
    {$sMemoTag}
    {$this->getXmlTagPresentation()}
</Payment>
XML;

        return $tag;
    }

    /**
     * returns the Memo xml tag
     *
     * @return null|string
     */
    public function getXmlTagMemo()
    {
        $sTestErrorCode  = $this->configuration->getTestErrorCode();
        $sTestReturnCode = $this->configuration->getTestReturnCode();
        $sCombined       = $sTestErrorCode . $sTestReturnCode;

        if (empty($sCombined)) {
            return null;
        }

        $sErrorCode  = '';
        $sReturnCode = '';
        if ($sTestErrorCode) {
            $sErrorCode = "error_code=$sTestErrorCode";
        }

        if ($sTestReturnCode) {
            $sReturnCode = ";return_code==$sTestReturnCode";
        }

        $tag = <<<XML
<Memo>{$sErrorCode}{$sReturnCode}</Memo>
XML;

        return $tag;
    }

    /**
     * returns the Presentation xml tag
     *
     * @return string
     */
    public function getXmlTagPresentation()
    {
        $tag = <<<XML
<Presentation>
    {$this->getXmlTagAmount()}
    {$this->getXmlTagCurrency()}
    {$this->getXmlTagUsage()}
</Presentation>
XML;

        return $tag;
    }

    /**
     * returns the Amount xml tag
     *
     * @return string
     */
    public function getXmlTagAmount()
    {
        $dAmount = $this->currentPaymentData->getAmount();
        $tag     = <<<XML
<Amount>{$dAmount}</Amount>
XML;

        return $tag;
    }

    /**
     * returns the Currency xml tag
     *
     * @return string
     * @throws \Exception
     */
    public function getXmlTagCurrency()
    {
        $currency = $this->currentPaymentData->getCurrency();
        $tag      = <<<XML
<Currency>{$currency}</Currency>
XML;

        return $tag;
    }

    /**
     * returns the Usage xml tag
     *
     * @return string
     */
    public function getXmlTagUsage()
    {
        return '<Usage/>';
    }

    /**
     * returns the Frontend xml tag
     *
     * @return string
     */
    public function getXmlTagFrontend()
    {
        $sElement = <<<XML
<Frontend>
    {$this->getXmlTagResponseUrl()}
    {$this->getXmlTagSessionID()}
</Frontend>
XML;

        return $sElement;
    }

    /**
     * returns the ResponseUrl xml tag
     *
     * @return string
     */
    public function getXmlTagResponseUrl()
    {
        $sResponseUrl = $this->configuration->getResponseUrl();

        $tag = <<<XML
<ResponseUrl>{$sResponseUrl}</ResponseUrl>
XML;

        return $tag;
    }

    /**
     * returns the SessionID xml tag
     *
     * @return string
     */
    public function getXmlTagSessionID()
    {
        $sessionId = $this->currentPaymentData->getSessionId();
        $tag       = <<<XML
<SessionID>{$sessionId}</SessionID>
XML;

        return $tag;
    }

    /**
     * returns the Account xml tag
     *
     * @return string
     * @throws invalidheidelpaydataexception
     */
    public function getXmlTagAccount()
    {
        $sPaymentCode = $this->currentPaymentData->payment->getPaymentCode();
        if ($this->currentPaymentData->payment instanceof billsafe || in_array($sPaymentCode, array('OT', 'VA'))) {
            return $this->getXMLAccountData();
        }

        $sIdentificationUniqueId = $this->currentPaymentData->getUniqueId();

        if (empty($sIdentificationUniqueId)) {
            $oEx = new invalidheidelpaydataexception('IDENTIFICATION_UNIQUEID');
            throw $oEx;
        }

        $tag = <<<XML
<Account registration="{$sIdentificationUniqueId}"/>
XML;

        return $tag;
    }

    /**
     * returns the result of the heidelpay payment generated method.
     * See at methods like getXMLAccountData_billsafe, getXMLAccountData_ideal, etc..
     *
     * @return mixed
     * @throws UnknownheidelpayonlinetransferpaymentException
     */
    public function getXMLAccountData()
    {
        $aClassParts = explode('\\', get_class($this->currentPaymentData->payment));
        $sOTSubType  = end($aClassParts);

        $sMethodName = "getXMLAccountData" . $sOTSubType;

        if (false == method_exists($this, $sMethodName)) {
            $oEx = new UnknownheidelpayonlinetransferpaymentException(get_class($this->currentPaymentData->payment));
            throw $oEx;
        }

        return $this->$sMethodName();
    }

    /**
     * Returns the ideal specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataIdeal()
    {
        $bankName    = $this->currentPaymentData->getBankName();
        $bankCountry = $this->currentPaymentData->getAccountCountry();
        $sXmlData    = <<<XML
<Account>
    <BankName>{$bankName}</BankName>
    <Country>{$bankCountry}</Country>
</Account>
XML;
        if ($this->configuration->isTestmode()) {
            $sXmlData = <<<XML
<Account>
    <BankName>ING_TEST</BankName>
    <Country>NL</Country>
</Account>
XML;
        }

        return $sXmlData;
    }

    /**
     * Returns the eps specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataEps()
    {
        $accountHolder = $this->currentPaymentData->getAccountHolder();
        $bankName      = $this->currentPaymentData->getBankName();
        $bankCountry   = $this->currentPaymentData->getAccountCountry();
        $sXmlData      = <<<XML
<Account>
    <Holder>{$accountHolder}</Holder>
    <BankName>{$bankName}</BankName>
    <Country>{$bankCountry}</Country>
</Account>
XML;

        if ($this->configuration->isTestmode()) {
            $sXmlData = <<<XML
<Account>
    <Holder>John Doe</Holder>
    <BankName>ERSTE_BANK_TEST</BankName>
    <Country>AT</Country>
</Account>
XML;
        }

        return $sXmlData;
    }

    /**
     * Returns the sofortueberweisung specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataSofortueberweisung()
    {
        return <<<XML
<Account>
    <Iban/>
    <Bic/>
    <Country/>
</Account>
XML;
    }

    /**
     * Returns the giropay specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataGiropay()
    {
        if ($this->currentPaymentData->isSepaMode()) {
            return $this->getXMLAccountDataGiropaySEPA();
        }

        return $this->getXMLAccountDataGiropayNonSEPA();
    }

    /**
     * Returns the giropay specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataGiropaySEPA()
    {
        if ($this->configuration->isTestmode()) {
            $sXmlData = <<<XML
<Account>
    <Iban>DE46940594210000012345</Iban>
    <Bic>TESTDETT421</Bic>
    <Country>DE</Country>
</Account>
XML;

            return $sXmlData;
        }

        $iban           = $this->currentPaymentData->getIban();
        $bic            = $this->currentPaymentData->getBic();
        $accountCountry = $this->currentPaymentData->getAccountCountry();

        $sXmlData = <<<XML
<Account>
    <Iban>{$iban}</Iban>
    <Bic>{$bic}</Bic>
    <Country>{$accountCountry}</Country>
</Account>
XML;

        return $sXmlData;
    }

    /**
     * Returns the giropay specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataGiropayNonSEPA()
    {
        if ($this->configuration->isTestmode()) {
            $sXmlData = <<<XML
<Account>
    <Number>0000000300</Number>
    <Holder>testname</Holder>
    <Bank>12345679</Bank>
    <Country>DE</Country>
</Account>
XML;

            return $sXmlData;
        }

        $accountNumber  = $this->currentPaymentData->getAccountNumber();
        $accountHolder  = $this->currentPaymentData->getAccountHolder();
        $bankCode       = $this->currentPaymentData->getBankCode();
        $accountCountry = $this->currentPaymentData->getAccountCountry();

        $sXmlData = <<<XML
<Account>
    <Number>{$accountNumber}</Number>
    <Holder>{$accountHolder}</Holder>
    <Bank>{$bankCode}</Bank>
    <Country>{$accountCountry}</Country>
</Account>
XML;

        return $sXmlData;
    }

    /**
     * Returns the billsafe specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataBillsafe()
    {

        $sXmlData = <<<XML
<Account>
    <Brand>BILLSAFE</Brand>
</Account>
XML;

        return $sXmlData;
    }

    /**
     * Returns the billsafe specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataPrzelewy24()
    {

        $sXmlData = <<<XML
<Account>
    <Brand>Przelewy24</Brand>
</Account>
XML;

        return $sXmlData;
    }

    /**
     * Returns the paypal specific account xml tag
     *
     * @return string
     */
    public function getXMLAccountDataPaypal()
    {
        $sXmlData = <<<XML
<Account>
    <Brand>PAYPAL</Brand>
</Account>
XML;

        return $sXmlData;
    }

    /**
     * Returns the Analysis xml tag
     *
     * @return string
     */
    public function getXmlTagAnalysis()
    {
        $tag = <<<XML
<Analysis/>
XML;

        return $tag;
    }

    /**
     * Returns the postfinance specific account xml tag
     *
     * @return string
     */
    //    public function getXMLAccountData_postfinance()
    //    {
    //        $sXmlData = <<<XML
    //<Account>
    //    <Brand>PFEFINANCE</Brand>
    //</Account>
    //XML;
    //
    //        return $sXmlData;
    //    }

    /**
     * Returns the Customer xml tag
     *
     * @return string
     */
    public function getXmlTagCustomer()
    {
        $tag = <<<XML
<Customer>
    {$this->getXmlTagName()}
    {$this->getXmlTagContact()}
    {$this->getXmlTagAddress()}
</Customer>
XML;

        return $tag;
    }

    /**
     * Returns the Name xml tag
     *
     * @return string
     */
    public function getXmlTagName()
    {
        $tag = <<<XML
<Name>
    {$this->getXmlTagFamily()}
    {$this->getXmlTagGiven()}
    {$this->getXmlTagCompany()}
    {$this->getXmlTagSalutation()}
    {$this->getXmlTagTitle()}
    {$this->getXmlTagSex()}
    {$this->getXmlTagBirthday()}
</Name>
XML;

        return $tag;
    }

    /**
     * Returns the Family xml tag
     *
     * @return string
     */
    public function getXmlTagFamily()
    {
        $familyName = $this->currentPaymentData->getUserFamilyName();
        $tag        = <<<XML
<Family>{$familyName}</Family>
XML;

        return $tag;
    }

    /**
     * Returns the Given xml tag
     *
     * @return string
     */
    public function getXmlTagGiven()
    {
        $firstName = $this->currentPaymentData->getUserFirstName();
        $tag       = <<<XML
<Given>{$firstName}</Given>
XML;

        return $tag;
    }

    /**
     * Returns the Company xml tag
     *
     * @return string
     */
    public function getXmlTagCompany()
    {
        $sField   = $this->currentPaymentData->getUserCompany();
        $sCompany = $sField;
        $tag      = <<<XML
<Company>{$sCompany}</Company>
XML;

        return $tag;
    }

    /**
     * Returns the Salutation xml tag
     *
     * @return string
     */
    public function getXmlTagSalutation()
    {
        $tag = <<<XML
<Salutation/>
XML;

        return $tag;
    }

    /**
     * Returns the Title xml tag
     *
     * @return string
     */
    public function getXmlTagTitle()
    {
        $tag = <<<XML
<Title/>
XML;

        return $tag;
    }

    /**
     * Returns the Sex xml tag
     *
     * @return string
     */
    public function getXmlTagSex()
    {
        $tag = <<<XML
<Sex/>
XML;

        return $tag;
    }

    /**
     * Returns the Birthdate xml tag
     *
     * @return string
     */
    public function getXmlTagBirthday()
    {
        $birthDate = $this->currentPaymentData->getUserBirthdate();
        $tag       = <<<XML
<Birthdate>{$birthDate}</Birthdate>
XML;

        return $tag;
    }

    /**
     * Returns the Contact xml tag
     *
     * @return string
     */
    public function getXmlTagContact()
    {
        $tag = <<<XML
<Contact>
    {$this->getXmlTagEmail()}
    {$this->getXmlTagIp()}
    {$this->getXmlTagMobile()}
    {$this->getXmlTagPhone()}
</Contact>
XML;

        return $tag;
    }

    /**
     * Returns the Email xml tag
     *
     * @return string
     */
    public function getXmlTagEmail()
    {
        $sEmail = $this->currentPaymentData->getUserEMailAddress();
        $tag    = <<<XML
<Email>{$sEmail}</Email>
XML;

        return $tag;
    }

    /**
     * Returns the Ip xml tag
     *
     * @return string
     */
    public function getXmlTagIp()
    {
        $sIp = $this->configuration->getRemoteAddress();
        $tag = <<<XML
<Ip>{$sIp}</Ip>
XML;

        return $tag;
    }

    /**
     * Returns the Mobile xml tag
     *
     * @return string
     */
    public function getXmlTagMobile()
    {
        $tag = <<<XML
<Mobile/>
XML;

        return $tag;
    }

    /**
     * Returns the Phone xml tag
     *
     * @return string
     */
    public function getXmlTagPhone()
    {
        $sField = $this->currentPaymentData->getUserPhone();
        $sPhone = $sField;
        $tag    = <<<XML
<Phone>{$sPhone}</Phone>
XML;

        return $tag;
    }

    /**
     * Returns the Address xml tag
     *
     * @return string
     */
    public function getXmlTagAddress()
    {
        $tag = <<<XML
<Address>
    {$this->getXmlTagCity()}
    {$this->getXmlTagCountry()}
    {$this->getXmlTagState()}
    {$this->getXmlTagStreet()}
    {$this->getXmlTagZip()}
</Address>
XML;

        return $tag;
    }

    /**
     * Returns the City xml tag
     *
     * @return string
     */
    public function getXmlTagCity()
    {
        $city = $this->currentPaymentData->getUserCity();
        $tag  = <<<XML
<City>{$city}</City>
XML;

        return $tag;
    }

    /**
     * Returns the Country xml tag
     *
     * @return string
     */
    public function getXmlTagCountry()
    {
        $countryIsoAlpha = $this->currentPaymentData->getUserCountryCode();
        $tag             = <<<XML
<Country>{$countryIsoAlpha}</Country>
XML;

        return $tag;
    }

    /**
     * Returns the State xml tag
     *
     * @return string
     */
    public function getXmlTagState()
    {
        $tag = <<<XML
<State/>
XML;

        return $tag;
    }

    /**
     * Returns the Street xml tag
     *
     * @return string
     */
    public function getXmlTagStreet()
    {
        $street = $this->currentPaymentData->getUserStreet();
        $tag    = <<<XML
<Street>{$street}</Street>
XML;

        return $tag;
    }

    /**
     * Returns the Zip xml tag
     *
     * @return string
     */
    public function getXmlTagZip()
    {
        $zip = $this->currentPaymentData->getUserZip();
        $tag = <<<XML
<Zip>{$zip}</Zip>
XML;

        return $tag;
    }

    /**
     * Returns the TestAdvise xml tag
     *
     * @return string
     */
    public function getXmlTagTestAdvise()
    {
        $testAdvise = "";
        if ($this->configuration->isTestmode()) {
            $testAdvise = "VALIDATE=false";
        }

        $tag = <<<XML
<TestAdvise>{$testAdvise}</TestAdvise>
XML;

        return $tag;
    }

    /**
     * Returns the complete request xml
     *
     * @return string
     */
    public function getXmlTagQuery()
    {
        $mode    = $this->configuration->getTransactionMode();
        $channel = $this->currentPaymentData->getPaymentChannel();
        $date    = $this->currentPaymentData->getSearchDate();

        $tag = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<Request version="1.0">
    {$this->getXmlTagHeader()}
    <Query mode="{$mode}" level="CHANNEL" entity="{$channel}" type="STANDARD">
        {$this->getXmlTagUser()}
        <Period from="{$date}" to="{$date}"/>
        <Types>
            <Type code="RC"/>
        </Types>
    </Query>
</Request>
XML;

        return $tag;
    }

    /**
     * Returns the Header xml tag
     *
     * @return string
     */
    public function getXmlTagHeader()
    {
        $tag = <<<XML
<Header>
    {$this->getXmlTagSecurity()}
</Header>
XML;

        return $tag;
    }

    /**
     * Returns the Security xml tag
     *
     * @return string
     * @throws emptysecuritysenderexception
     */
    public function getXmlTagSecurity()
    {
        $securitySender = $this->configuration->getSecuritySender();

        $tag = <<<XML
<Security sender="{$securitySender}"/>
XML;

        return $tag;
    }

    /**
     * Returns the user-xml-tag.
     *
     * @return string
     * @throws emptypasswordexception
     * @throws emptyusernameexception
     */
    public function getXmlTagUser()
    {
        $userId   = $this->configuration->getUserID();
        $password = $this->configuration->getPassword();

        $tag = <<<XML
<User login="{$userId}" pwd="{$password}"/>
XML;

        return $tag;
    }

    /**
     * Returns the Analysis>Criterion xml tags if available
     *
     * @return string
     * @throws NoBasketException
     */
    public function getXmlTagCriterion()
    {
        $sPaymentCode = $this->currentPaymentData->payment->getPaymentCode();
        if (false == ($this->currentPaymentData->payment instanceof Billsafe) && $sPaymentCode != 'IV') {
            return '';
        }

        $sXML = "<Analysis>" . PHP_EOL;
        $sXML .= $this->getCriterionContents();
        $sXML .= "</Analysis>" . PHP_EOL;

        return $sXML;
    }

    /**
     * Returns the Criterion xml tags filled with basket contents
     *
     * @return string
     */
    protected function getCriterionContents()
    {
        $criterionCounter = 1;
        $XML              = "";
        foreach ($this->currentPaymentData->getCriterionPositions() as $position) {
            /** @var Position $position */
            $iPos = sprintf("%02d", $criterionCounter);
            $XML .= $this->getCriterionTag("POS_{$iPos}.POSITION", $criterionCounter);
            $XML .= $this->getCriterionTag("POS_{$iPos}.PERCENT_VAT", $position->getPercentVat());
            $XML .= $this->getCriterionTag("POS_{$iPos}.AMOUNT_UNIT_GROSS", $position->getAmountUnitGross());
            $XML .= $this->getCriterionTag("POS_{$iPos}.AMOUNT_GROSS", $position->getAmountGross());
            $XML .= $this->getCriterionTag("POS_{$iPos}.ARTICLE_TYPE", $position->getType());
            $XML .= $this->getCriterionTag("POS_{$iPos}.ARTICLE_NUMBER", $position->getNumber());
            $XML .= $this->getCriterionTag("POS_{$iPos}.TEXT", $position->getText());
            $XML .= $this->getCriterionTag("POS_{$iPos}.UNIT", $position->getUnit());
            $XML .= $this->getCriterionTag("POS_{$iPos}.QUANTITY", $position->getQuantity());
            $criterionCounter++;
        }

        return $XML;
    }

    protected function getCriterionTag($name, $value)
    {
        return "<Criterion name='{$name}'>{$value}</Criterion>" . PHP_EOL;
    }

    /**
     * Generates a xml via given pattern.
     *
     * @param pattern\AbstractPattern $oPattern
     * @param payment\Data            $paymentData
     *
     * @return string
     * @throws InvalidTagNameException
     */
    public function generateForPattern(pattern\AbstractPattern $oPattern, payment\Data $paymentData)
    {
        $this->currentPaymentData = $paymentData;
        $sPattern                 = $oPattern->getPattern();
        $aTags                    = explode("\n", $sPattern);

        $aTagContents = array();
        foreach ($aTags as $tag) {
            if ('Query' == $tag) {
                return $this->getXmlTag($tag);
            }
            if (strlen(trim($tag))) {
                $aTagContents[] = $this->getXmlTag($tag);
            }
        }

        return str_replace(
            self::XML_BODY_PLACEHOLDER,
            implode("\n", $aTagContents),
            $this->getRequestXmlEnvelope()
        );
    }

    /**
     * Hookmethod for getXmlTagXXX
     *
     * @param $tag
     *
     * @return string
     * @throws InvalidTagNameException
     */
    public function getXmlTag($tag)
    {
        $sMethod = "getXmlTag$tag";
        if (false == method_exists($this, $sMethod)) {
            $oEx = new invalidtagnameException($tag);
            throw $oEx;
        }

        return call_user_func(array($this, $sMethod));
    }

    /**
     * Returns the enveloped request xml
     *
     * @return string
     */
    public function getRequestXmlEnvelope()
    {
        $trransactionMode = $this->configuration->getTransactionMode();
        $channel          = $this->currentPaymentData->getPaymentChannel();

        $sEnvelope = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<Request version="1.0">
    {$this->getXmlTagHeader()}
    <Transaction mode="{$trransactionMode}" response="SYNC" source="XML" channel="{$channel}">
        XML_BODY_PLACEHOLDER
    </Transaction>
</Request>
XML;

        return str_replace('XML_BODY_PLACEHOLDER', self::XML_BODY_PLACEHOLDER, $sEnvelope);
    }
}
